<?php
/**
 * User Has Tag
 *
 * @package     AutomatorWP\Integrations\Groundhogg\Filters\User_Has_Tag
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_Groundhogg_User_Has_Tag_Filter extends AutomatorWP_Integration_Filter {

    public $integration = 'groundhogg';
    public $filter = 'groundhogg_user_has_tag';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_filter( $this->filter, array(
            'integration'       => $this->integration,
            'label'             => __( 'User has tag', 'automatorwp-pro' ),
            'select_option'     => __( 'User has <strong>tag</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Condition. %2$s: Tag. */
            'edit_label'        => sprintf( __( '%1$s %2$s', 'automatorwp-pro' ), '{condition}', '{tag}' ),
            /* translators: %1$s: Condition. %2$s: Tag. */
            'log_label'         => sprintf( __( '%1$s %2$s', 'automatorwp-pro' ), '{condition}', '{tag}'),
            'options'           => array(
                'tag' => automatorwp_utilities_ajax_selector_option( array(
                    'field'             => 'tag',
                    'option_default'    => __( 'Select a tag', 'automatorwp-pro' ),
                    'name'              => __( 'Tag:', 'automatorwp-pro' ),
                    'option_custom'         => true,
                    'option_custom_desc'    => __( 'Tag ID', 'automatorwp-pro' ),
                    'action_cb'         => 'automatorwp_groundhogg_get_tags',
                    'options_cb'        => 'automatorwp_groundhogg_options_cb_tag',
                    'default'           => ''
                ) ),
                'condition' => array(
                    'from' => 'condition',
                    'fields' => array(
                        'condition' => array(
                            'name' => __( 'Condition:', 'automatorwp-pro' ),
                            'type' => 'select',
                            'options' => array(
                                'equal'             => __( 'equal to', 'automatorwp-pro' ),
                                'not_equal'         => __( 'not equal to', 'automatorwp-pro' ),
                            ),
                            'default' => 'equal'
                        )
                    )
                ),
            ),
        ) );

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_filter    True if user deserves filter, false otherwise
     * @param stdClass  $filter             The filter object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $filter_options     The filter's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_filter( $deserves_filter, $filter, $user_id, $event, $filter_options, $automation ) {

        global $wpdb;
        
        // Shorthand
        $tag_id = absint( $filter_options['tag'] );
        $condition = $filter_options['condition'];

        // Bail if empty tag to assign
        if( empty( $tag_id ) ) {
            return;
        }

        $tag = Groundhogg\Plugin::$instance->dbs->get_db( 'tags' )->get( $tag_id );

        // Bail if tag not exists
        if( ! $tag ) {
            $this->result = __( 'Filter not passed. Tag does not exist.', 'automatorwp-pro' );
            return false;
        }

        $user_data = get_user_by('ID', $user_id );
        $contact = new Groundhogg\Contact( $user_data->user_email, false );

        // Bail if contact not exists
        if( ! $contact ) {
            $this->result = __( 'Filter not passed. User is not a contact.', 'automatorwp-pro' );
            return false;
        }

        switch( $condition ) {
            case 'equal':
                // Don't deserve if user does not have the tag
                if( ! $contact->has_tag( $tag->tag_id ) ) {
                    $this->result = sprintf( __( 'Filter not passed. User does not meet the condition tag %1$s "%2$s".', 'automatorwp-pro' ),
                        automatorwp_utilities_get_condition_label( $condition ),
                        $tag->tag_name
                    );
                    return false;
                }
                break;
            case 'not_equal':
                // Don't deserve if user has the tag
                if( $contact->has_tag( $tag->tag_id ) ) {
                    $this->result = sprintf( __( 'Filter not passed. User does not meet the condition tag %1$s "%2$s".', 'automatorwp-pro' ),
                    automatorwp_utilities_get_condition_label( $condition ),
                    $tag->tag_name
                    );
                    return false;
                }
                break;
        }

        $this->result = sprintf( __( 'Filter passed. User has tag %1$s "%2$s".', 'automatorwp-pro' ),
            automatorwp_utilities_get_condition_label( $condition ),
            $tag->tag_name
        );

        return $deserves_filter;

    }

}

new AutomatorWP_Groundhogg_User_Has_Tag_Filter();